using System.Collections.Generic;
using UnityEngine;
using System.Linq;

/// <summary>
/// Controls the sensor graph visualization, including updating sensor data,
/// rendering graph lines, and managing legend entries.
/// </summary>
public class SensorGraphController : MonoBehaviour
{
    /// <summary>
    /// Prefab for the graph line.
    /// </summary>
    [Header("Graph Prefabs")]
    [SerializeField] private GameObject linePrefab;

    /// <summary>
    /// Prefab for the graph points.
    /// </summary>
    [SerializeField] private GameObject pointPrefab;

    /// <summary>
    /// Prefab for the reference lines on the Y-axis.
    /// </summary>
    [SerializeField] private GameObject referenceLinePrefab;

    /// <summary>
    /// Parent transform for the reference lines.
    /// </summary>
    [SerializeField] private Transform referenceLineParent;

    /// <summary>
    /// Prefab for the legend items.
    /// </summary>
    [SerializeField] private GameObject legendItemPrefab;

    /// <summary>
    /// Parent transform for the legend items.
    /// </summary>
    [SerializeField] private Transform legendParent;

    /// <summary>
    /// Minimum value for the Y-axis.
    /// </summary>
    [Header("Graph Settings")]
    [SerializeField] private float yMin = -2f;

    /// <summary>
    /// Maximum value for the Y-axis.
    /// </summary>
    [SerializeField] private float yMax = 45f;

    /// <summary>
    /// Desired maximum height for the graph.
    /// </summary>
    [SerializeField] private float desiredMaxHeight = 1f;

    /// <summary>
    /// Vertical offset for the graph.
    /// </summary>
    [SerializeField] private float verticalGraphOffset = -0.65f;

    /// <summary>
    /// Horizontal offset applied to the graph points.
    /// Determines the spacing between consecutive points on the graph.
    /// </summary>
    [SerializeField] private float horizontalGraphOffset = 0.31f;

    /// <summary>
    /// Step size for drawing reference lines on the Y-axis.
    /// Defines the interval between each reference line.
    /// </summary>
    [SerializeField] private float referenceLinesStep = 5f;

    /// <summary>
    /// Represents a mapping between sensor types and their associated colors.
    /// </summary>
    [System.Serializable]
    public class SensorColorEntry
    {
        /// <summary>
        /// The type of the sensor.
        /// </summary>
        public string sensorType;

        /// <summary>
        /// The color associated with the sensor type.
        /// </summary>
        public Color color;
    }

    /// <summary>
    /// List of sensor types and their associated colors.
    /// </summary>
    [SerializeField] private List<SensorColorEntry> sensorColors;

    /// <summary>
    /// Dictionary mapping sensor types to their corresponding line renderers.
    /// </summary>
    private readonly Dictionary<string, LineRenderer> _sensorLines = new();

    /// <summary>
    /// Dictionary mapping sensor types to their recent values.
    /// </summary>
    private readonly Dictionary<string, Queue<float>> _sensorValues = new();

    /// <summary>
    /// Maximum number of values to store for each sensor.
    /// </summary>
    private const int MaxValues = 10;

    /// <summary>
    /// Dictionary mapping sensor types to their graph points.
    /// </summary>
    private readonly Dictionary<string, List<GameObject>> _sensorPoints = new();

    /// <summary>
    /// Set of sensor types that have legend entries.
    /// </summary>
    private readonly HashSet<string> _legendEntries = new();

    /// <summary>
    /// Initializes the Y-axis reference lines when the script starts.
    /// </summary>
    private void Start() => DrawYAxisReferenceLines();

    /// <summary>
    /// Updates the sensor data and its corresponding graph visualization.
    /// If the sensor type is new, initializes its data structures and visual elements.
    /// </summary>
    /// <param name="sensorType">The type of the sensor being updated.</param>
    /// <param name="newValue">The new value to add to the sensor's data.</param>
    public void UpdateSensor(string sensorType, float newValue)
    {
        if (!_sensorValues.TryGetValue(sensorType, out var values))
        {
            values = new Queue<float>();
            _sensorValues[sensorType] = values;

            _sensorLines[sensorType] = CreateLine(sensorType);

            if (_legendEntries.Add(sensorType))
                CreateLegendEntry(sensorType);

            _sensorPoints[sensorType] = new List<GameObject>();
        }

        if (values.Count >= MaxValues) values.Dequeue();
        values.Enqueue(newValue);

        UpdateLine(sensorType, values.ToList());
    }

    /// <summary>
    /// Creates a line renderer for the specified sensor type.
    /// Configures the line's color based on the sensor type.
    /// </summary>
    /// <param name="sensorType">The type of the sensor for which the line is created.</param>
    /// <returns>The created LineRenderer object.</returns>
    private LineRenderer CreateLine(string sensorType)
    {
        var line = Instantiate(linePrefab, transform).GetComponent<LineRenderer>();

        var colorEntry = sensorColors.FirstOrDefault(e => e.sensorType == sensorType);

        if (colorEntry != null)
        {
            line.startColor = line.endColor = colorEntry.color;
            line.material = new Material(line.material) { color = colorEntry.color };
        }

        return line;
    }

    /// <summary>
    /// Creates a legend entry for the specified sensor type.
    /// Configures the legend item's color and text based on the sensor type.
    /// </summary>
    /// <param name="sensorType">The type of the sensor for which the legend entry is created.</param>
    private void CreateLegendEntry(string sensorType)
    {
        var item = Instantiate(legendItemPrefab, legendParent);

        var colorEntry = sensorColors.FirstOrDefault(e => e.sensorType == sensorType);
        if (colorEntry != null)
        {
            var cube = item.transform.Find("ColorIndicator")?.GetComponent<Renderer>();
            if (cube != null) cube.material.color = colorEntry.color;

            var text = item.transform.Find("NameText")?.GetComponent<TMPro.TextMeshPro>();
            if (text != null) text.text = sensorType;
        }
    }

    /// <summary>
    /// Updates the graph line and points for the specified sensor type.
    /// Adjusts the positions of the points based on the provided sensor values.
    /// </summary>
    /// <param name="sensorType">The type of the sensor being updated.</param>
    /// <param name="values">The list of values to visualize on the graph.</param>
    private void UpdateLine(string sensorType, List<float> values)
    {
        var line = _sensorLines[sensorType];
        var scaleFactor = desiredMaxHeight / Mathf.Max(0.0001f, yMax - yMin);

        line.positionCount = values.Count;

        ClearSensorPoints(sensorType);

        var colorEntry = sensorColors.FirstOrDefault(e => e.sensorType == sensorType);
        var color = colorEntry?.color;

        for (int i = 0; i < values.Count; i++)
        {
            var pos = new Vector3(-1.4f + i * horizontalGraphOffset, (values[i] - yMin) * scaleFactor + verticalGraphOffset, 0);
            line.SetPosition(i, pos);

            var point = CreateSensorPoint(sensorType, pos, color);
            _sensorPoints[sensorType].Add(point);
        }
    }

    /// <summary>
    /// Clears all existing graph points for the specified sensor type.
    /// </summary>
    /// <param name="sensorType">The type of the sensor whose points are being cleared.</param>
    private void ClearSensorPoints(string sensorType)
    {
        _sensorPoints[sensorType].ForEach(Destroy);
        _sensorPoints[sensorType].Clear();
    }

    /// <summary>
    /// Creates a new graph point for the specified sensor type.
    /// Configures the point's position and color.
    /// </summary>
    /// <param name="sensorType">The type of the sensor for which the point is created.</param>
    /// <param name="position">The position of the point in the graph.</param>
    /// <param name="color">The color of the point, if applicable.</param>
    /// <returns>The created graph point GameObject.</returns>
    private GameObject CreateSensorPoint(string sensorType, Vector3 position, Color? color)
    {
        var point = Instantiate(pointPrefab, _sensorLines[sensorType].transform);
        point.transform.localPosition = position;

        var pointRenderer = point.GetComponent<Renderer>();
        if (pointRenderer != null && color != null)
        {
            pointRenderer.material.color = color.Value;
        }

        return point;
    }

    /// <summary>
    /// Draws reference lines on the Y-axis based on the configured range and step size.
    /// </summary>
    private void DrawYAxisReferenceLines()
    {
        if (referenceLinePrefab == null || referenceLineParent == null) return;

        referenceLineParent.Cast<Transform>().ToList().ForEach(child => Destroy(child.gameObject));

        float scaleFactor = desiredMaxHeight / (yMax - yMin);

        for (float y = yMin; y <= yMax; y += referenceLinesStep)
        {
            var line = Instantiate(referenceLinePrefab, referenceLineParent);
            line.transform.localPosition = new Vector3(0, (y - yMin) * scaleFactor + verticalGraphOffset, 0);

            var text = line.GetComponentInChildren<TMPro.TextMeshPro>();
            if (text != null) text.text = Mathf.RoundToInt(y).ToString();
        }
    }
}